/*
 * Decompiled with CFR 0.152.
 */
package net.filebot.util;

import com.ibm.icu.text.CharsetDetector;
import com.ibm.icu.text.CharsetMatch;
import java.io.BufferedInputStream;
import java.io.File;
import java.io.FileFilter;
import java.io.FileInputStream;
import java.io.FilenameFilter;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.Reader;
import java.math.BigInteger;
import java.nio.ByteBuffer;
import java.nio.channels.FileChannel;
import java.nio.charset.Charset;
import java.nio.charset.StandardCharsets;
import java.nio.file.AtomicMoveNotSupportedException;
import java.nio.file.FileVisitOption;
import java.nio.file.FileVisitResult;
import java.nio.file.FileVisitor;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.nio.file.SimpleFileVisitor;
import java.nio.file.StandardCopyOption;
import java.nio.file.StandardOpenOption;
import java.nio.file.attribute.BasicFileAttributes;
import java.nio.file.attribute.FileAttribute;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.Comparator;
import java.util.EnumSet;
import java.util.HashMap;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.SortedMap;
import java.util.TreeMap;
import java.util.TreeSet;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.stream.Stream;
import net.filebot.Logging;
import net.filebot.util.AlphanumComparator;
import net.filebot.util.BOM;
import net.filebot.util.RegularExpressions;
import org.apache.commons.io.FileUtils;

public final class FileUtilities {
    private static final String WIN_THUMBNAIL_STORE = "Thumbs.db";
    private static final String MAC_THUMBNAIL_STORE = ".DS_Store";
    public static final Pattern EXTENSION = Pattern.compile("(?<=.[.])[\\p{Alnum}-]+$");
    public static final String UNC_PREFIX = "\\\\";
    public static final int FILE_WALK_MAX_DEPTH = 32;
    public static final Pattern ILLEGAL_CHARACTERS = Pattern.compile("[\\\\/:*?\"<>|\\r\\n]|\\p{Cntrl}|\\s+$|(?<=[^.])[.]+$|(?<=.{250})(.+)(?=[.]\\p{Alnum}{3}$)");
    public static final int BUFFER_SIZE = 65536;
    public static final long ONE_KILOBYTE = 1000L;
    public static final long ONE_MEGABYTE = 1000000L;
    public static final long ONE_GIGABYTE = 1000000000L;
    public static final FileFilter FOLDERS = File::isDirectory;
    public static final FileFilter FILES = File::isFile;
    public static final FileFilter NOT_HIDDEN = FileUtilities.not(File::isHidden);
    public static final FileFilter TEMPORARY = new FileFilter(){
        private final String tmpdir = System.getProperty("java.io.tmpdir");

        @Override
        public boolean accept(File file) {
            return file.getPath().startsWith(this.tmpdir);
        }
    };
    public static final Comparator<File> CASE_INSENSITIVE_PATH_ORDER = Comparator.comparing(File::getPath, String.CASE_INSENSITIVE_ORDER);
    public static final Comparator<File> HUMAN_NAME_ORDER = Comparator.comparing(File::getName, new AlphanumComparator(Locale.ENGLISH));

    public static File moveRename(File source, File destination) throws IOException {
        if (FileUtilities.equalsCaseSensitive(source, destination = FileUtilities.resolveDestination(source, destination))) {
            return destination;
        }
        if (source.isDirectory()) {
            FileUtils.moveDirectory(source, destination);
            return destination;
        }
        if (source.equals(destination)) {
            try {
                return Files.move(source.toPath(), destination.toPath(), StandardCopyOption.ATOMIC_MOVE).toFile();
            }
            catch (AtomicMoveNotSupportedException e) {
                Logging.debug.warning(e::toString);
            }
        }
        return Files.move(source.toPath(), destination.toPath(), StandardCopyOption.REPLACE_EXISTING).toFile();
    }

    public static File copyAs(File source, File destination) throws IOException {
        destination = FileUtilities.resolveDestination(source, destination);
        if (source.isDirectory()) {
            FileUtils.copyDirectory(source, destination);
            return destination;
        }
        return Files.copy(source.toPath(), destination.toPath(), StandardCopyOption.REPLACE_EXISTING).toFile();
    }

    public static File resolve(File source, File destination) {
        if (!destination.isAbsolute()) {
            destination = new File(source.getParentFile(), destination.getPath());
        }
        return destination;
    }

    public static File resolveDestination(File source, File destination) throws IOException {
        destination = FileUtilities.resolve(source, destination);
        Path parentFolder = destination.toPath().getParent();
        if (Files.notExists(parentFolder, LinkOption.NOFOLLOW_LINKS)) {
            Files.createDirectories(parentFolder, new FileAttribute[0]);
        }
        return destination;
    }

    public static File createRelativeSymlink(File link, File target, boolean relativize) throws IOException {
        if (relativize) {
            try {
                target = link.toPath().getParent().toRealPath(LinkOption.NOFOLLOW_LINKS).relativize(target.toPath()).toFile();
            }
            catch (Throwable e) {
                Logging.log.warning(Logging.cause("Unable to relativize link target", e));
            }
        }
        return Files.createSymbolicLink(link.toPath(), target.toPath(), new FileAttribute[0]).toFile();
    }

    public static File createHardLinkStructure(File link, File target) throws IOException {
        if (target.isFile()) {
            return Files.createLink(link.toPath(), target.toPath()).toFile();
        }
        final Path source = target.getCanonicalFile().toPath();
        final Path destination = link.getCanonicalFile().toPath();
        Files.walkFileTree(source, EnumSet.of(FileVisitOption.FOLLOW_LINKS), 32, (FileVisitor<? super Path>)new SimpleFileVisitor<Path>(){

            @Override
            public FileVisitResult visitFile(Path file, BasicFileAttributes attrs) throws IOException {
                Path linkFile = destination.resolve(source.relativize(file));
                Files.createDirectories(linkFile.getParent(), new FileAttribute[0]);
                Files.createLink(linkFile, file);
                return FileVisitResult.CONTINUE;
            }
        });
        return destination.toFile();
    }

    public static void delete(File file) throws IOException {
        if (file.isDirectory()) {
            Files.walkFileTree(file.toPath(), (FileVisitor<? super Path>)new SimpleFileVisitor<Path>(){

                @Override
                public FileVisitResult visitFile(Path f, BasicFileAttributes attr) throws IOException {
                    Files.delete(f);
                    return FileVisitResult.CONTINUE;
                }

                @Override
                public FileVisitResult postVisitDirectory(Path f, IOException e) throws IOException {
                    Files.delete(f);
                    return FileVisitResult.CONTINUE;
                }
            });
        } else {
            Files.delete(file.toPath());
        }
    }

    public static void createFolders(File folder) throws IOException {
        Files.createDirectories(folder.toPath(), new FileAttribute[0]);
    }

    public static boolean isThumbnailStore(File file) {
        return MAC_THUMBNAIL_STORE.equals(file.getName()) || WIN_THUMBNAIL_STORE.equalsIgnoreCase(file.getName());
    }

    public static byte[] readFile(File file) throws IOException {
        return Files.readAllBytes(file.toPath());
    }

    public static String readTextFile(File file) throws IOException {
        long size = file.length();
        if (size > 1000000000L) {
            throw new IOException(String.format("Text file is too large: %s (%s)", file, FileUtilities.formatSize(size)));
        }
        byte[] bytes = FileUtilities.readFile(file);
        BOM bom = BOM.detect(bytes);
        if (bom != null) {
            return new String(bytes, bom.size(), bytes.length - bom.size(), bom.getCharset());
        }
        return new String(bytes, StandardCharsets.UTF_8);
    }

    public static List<String> readLines(File file) throws IOException {
        return Arrays.asList(RegularExpressions.NEWLINE.split(FileUtilities.readTextFile(file)));
    }

    public static File writeFile(ByteBuffer data, File destination) throws IOException {
        try (FileChannel channel = FileChannel.open(destination.toPath(), StandardOpenOption.CREATE, StandardOpenOption.TRUNCATE_EXISTING, StandardOpenOption.WRITE);){
            channel.write(data);
        }
        return destination;
    }

    public static Reader createTextReader(InputStream in, boolean guess, Charset declaredEncoding) throws IOException {
        byte[] head = new byte[4];
        in.mark(head.length);
        in.read(head);
        in.reset();
        BOM bom = BOM.detect(head);
        if (bom != null) {
            in.skip(bom.size());
            return new InputStreamReader(in, bom.getCharset());
        }
        if (guess) {
            CharsetDetector detector = new CharsetDetector();
            detector.setDeclaredEncoding(declaredEncoding.name());
            detector.setText(in);
            CharsetMatch match = detector.detect();
            if (match != null) {
                Reader reader = match.getReader();
                if (reader != null) {
                    return reader;
                }
                switch (match.getName()) {
                    case "ISO-8859-8-I": {
                        return new InputStreamReader(in, Charset.forName("ISO-8859-8"));
                    }
                }
                Logging.debug.warning("Unsupported charset: " + match.getName());
            }
        }
        return new InputStreamReader(in, declaredEncoding);
    }

    public static Reader createTextReader(File file) throws IOException {
        return FileUtilities.createTextReader(new BufferedInputStream(new FileInputStream(file), 65536), true, StandardCharsets.UTF_8);
    }

    public static boolean equalsCaseSensitive(File a, File b) {
        return a.getPath().equals(b.getPath());
    }

    public static boolean equalsFileContent(File a, File b) {
        if (a.length() != b.length()) {
            return false;
        }
        if (a.isDirectory() || b.isDirectory()) {
            return false;
        }
        try {
            return FileUtils.contentEquals(a, b);
        }
        catch (Exception e) {
            Logging.log.warning(Logging.cause(e));
            return false;
        }
    }

    public static String getExtension(File file) {
        if (file.isDirectory()) {
            return null;
        }
        return FileUtilities.getExtension(file.getName());
    }

    public static String getExtension(String name) {
        Matcher matcher = EXTENSION.matcher(name);
        if (matcher.find()) {
            return matcher.group();
        }
        return null;
    }

    public static boolean hasExtension(File file, String ... extensions) {
        return FileUtilities.hasExtension(file.getName(), extensions) && !file.isDirectory();
    }

    public static boolean hasExtension(String filename, String ... extensions) {
        for (String it : extensions) {
            String tail;
            if (filename.length() - it.length() < 2 || filename.charAt(filename.length() - it.length() - 1) != '.' || !(tail = filename.substring(filename.length() - it.length(), filename.length())).equalsIgnoreCase(it)) continue;
            return true;
        }
        return false;
    }

    public static String getNameWithoutExtension(String name) {
        Matcher matcher = EXTENSION.matcher(name);
        if (matcher.find()) {
            return name.substring(0, matcher.start() - 1);
        }
        return name;
    }

    public static String getName(File file) {
        if (file == null) {
            return null;
        }
        if (file.isDirectory()) {
            return FileUtilities.getFolderName(file);
        }
        return FileUtilities.getNameWithoutExtension(file.getName());
    }

    public static String getFolderName(File file) {
        if (UNC_PREFIX.equals(file.getParent())) {
            return file.toString();
        }
        String name = file.getName();
        if (name.length() > 0) {
            return name;
        }
        return FileUtilities.replacePathSeparators(file.toString(), "");
    }

    public static boolean isDerived(File derivate, File prime) {
        String n = FileUtilities.getName(prime).toLowerCase();
        String s = FileUtilities.getName(derivate).toLowerCase();
        if (s.startsWith(n)) {
            return s.length() == n.length() || !Character.isLetterOrDigit(s.charAt(n.length()));
        }
        return false;
    }

    public static boolean isDerivedByExtension(File derivate, File prime) {
        return FileUtilities.isDerivedByExtension(FileUtilities.getName(derivate), prime);
    }

    public static boolean isDerivedByExtension(String derivate, File prime) {
        String base = FileUtilities.getName(prime).trim().toLowerCase();
        if ((derivate = derivate.trim().toLowerCase()).equals(base)) {
            return true;
        }
        while (derivate.length() > base.length() && FileUtilities.getExtension(derivate) != null) {
            if (!(derivate = FileUtilities.getNameWithoutExtension(derivate)).equals(base)) continue;
            return true;
        }
        return false;
    }

    public static boolean containsOnly(Collection<File> files, FileFilter filter) {
        if (files.isEmpty()) {
            return false;
        }
        for (File file : files) {
            if (filter.accept(file)) continue;
            return false;
        }
        return true;
    }

    public static List<File> sortByUniquePath(Collection<File> files) {
        TreeSet<File> sortedSet = new TreeSet<File>(CASE_INSENSITIVE_PATH_ORDER);
        sortedSet.addAll(files);
        return new ArrayList<File>(sortedSet);
    }

    public static List<File> filter(Iterable<File> files, FileFilter ... filters) {
        ArrayList<File> accepted = new ArrayList<File>();
        block0: for (File file : files) {
            for (FileFilter filter : filters) {
                if (!filter.accept(file)) continue;
                accepted.add(file);
                continue block0;
            }
        }
        return accepted;
    }

    public static FileFilter not(FileFilter filter) {
        return f -> !filter.accept(f);
    }

    public static FileFilter filter(FileFilter ... filters) {
        return f -> Arrays.stream(filters).anyMatch(it -> it.accept(f));
    }

    public static List<File> listPath(File file) {
        return FileUtilities.listPathTail(file, Integer.MAX_VALUE, false);
    }

    public static List<File> listPathTail(File file, int tailSize, boolean reverse) {
        LinkedList<File> nodes = new LinkedList<File>();
        File node = file;
        for (int i = 0; node != null && i < tailSize && !UNC_PREFIX.equals(node.toString()); ++i, node = node.getParentFile()) {
            if (reverse) {
                nodes.addLast(node);
                continue;
            }
            nodes.addFirst(node);
        }
        return nodes;
    }

    public static File getRelativePathTail(File file, int tailSize) {
        File f = null;
        for (File it : FileUtilities.listPathTail(file, tailSize, false)) {
            if (it.getParentFile() == null) continue;
            f = new File(f, it.getName());
        }
        return f;
    }

    public static List<File> getFileSystemRoots() {
        File[] roots = File.listRoots();
        if (roots == null) {
            roots = new File[]{};
        }
        return Arrays.asList(roots);
    }

    public static List<File> getChildren(File folder) {
        return FileUtilities.getChildren(folder, null, null);
    }

    public static List<File> getChildren(File folder, FileFilter filter) {
        return FileUtilities.getChildren(folder, filter, null);
    }

    public static List<File> getChildren(File folder, FileFilter filter, Comparator<File> order) {
        File[] files;
        File[] fileArray = files = filter == null ? folder.listFiles() : folder.listFiles(filter);
        if (files == null) {
            return Collections.emptyList();
        }
        if (order != null) {
            Arrays.sort(files, order);
        }
        return Arrays.asList(files);
    }

    public static List<File> listFiles(File folder, FileFilter filter) {
        return FileUtilities.listFiles(new File[]{folder}, 32, filter, null);
    }

    public static List<File> listFiles(File folder, FileFilter filter, Comparator<File> order) {
        return FileUtilities.listFiles(new File[]{folder}, 32, filter, order);
    }

    public static List<File> listFiles(Collection<File> folders, FileFilter filter, Comparator<File> order) {
        return FileUtilities.listFiles(folders.toArray(new File[0]), 32, filter, order);
    }

    public static List<File> listFiles(File[] files, int depth, FileFilter filter, Comparator<File> order) {
        ArrayList<File> sink = new ArrayList<File>();
        FileUtilities.streamFiles(files, FOLDERS, order).forEach(f -> FileUtilities.listFiles(f, sink, depth, filter, order));
        FileUtilities.streamFiles(files, filter, order).forEach(sink::add);
        return sink;
    }

    private static void listFiles(File folder, List<File> sink, int depth, FileFilter filter, Comparator<File> order) {
        if (depth < 0) {
            return;
        }
        File[] files = folder.listFiles(NOT_HIDDEN);
        FileUtilities.streamFiles(files, FOLDERS, order).forEach(f -> FileUtilities.listFiles(f, sink, depth - 1, filter, order));
        FileUtilities.streamFiles(files, filter, order).forEach(sink::add);
    }

    private static Stream<File> streamFiles(File[] files, FileFilter filter, Comparator<File> order) {
        if (files == null || files.length == 0) {
            return Stream.empty();
        }
        if (order == null) {
            return Arrays.stream(files).filter(filter::accept);
        }
        return Arrays.stream(files).filter(filter::accept).sorted(order);
    }

    public static SortedMap<File, List<File>> mapByFolder(Iterable<File> files) {
        TreeMap<File, List<File>> map = new TreeMap<File, List<File>>();
        for (File file : files) {
            File key = file.getParentFile();
            if (key == null) {
                throw new IllegalArgumentException("Parent is null: " + file);
            }
            ArrayList<File> valueList = (ArrayList<File>)map.get(key);
            if (valueList == null) {
                valueList = new ArrayList<File>();
                map.put(key, valueList);
            }
            valueList.add(file);
        }
        return map;
    }

    public static Map<String, List<File>> mapByExtension(Iterable<File> files) {
        HashMap<String, List<File>> map = new HashMap<String, List<File>>();
        for (File file : files) {
            ArrayList<File> valueList;
            String key = FileUtilities.getExtension(file);
            if (key != null) {
                key = key.toLowerCase();
            }
            if ((valueList = (ArrayList<File>)map.get(key)) == null) {
                valueList = new ArrayList<File>();
                map.put(key, valueList);
            }
            valueList.add(file);
        }
        return map;
    }

    public static String validateFileName(CharSequence filename) {
        return RegularExpressions.SPACE.matcher(ILLEGAL_CHARACTERS.matcher(filename).replaceAll("")).replaceAll(" ").trim();
    }

    public static boolean isInvalidFileName(CharSequence filename) {
        return ILLEGAL_CHARACTERS.matcher(filename).find();
    }

    public static File validateFileName(File file) {
        if (!FileUtilities.isInvalidFileName(file.getName())) {
            return file;
        }
        return new File(file.getParentFile(), FileUtilities.validateFileName(file.getName()));
    }

    public static File validateFilePath(File path) {
        Iterator<File> nodes = FileUtilities.listPath(path).iterator();
        File validatedPath = FileUtilities.validateFileName(nodes.next());
        while (nodes.hasNext()) {
            validatedPath = new File(validatedPath, FileUtilities.validateFileName(nodes.next().getName()));
        }
        return validatedPath;
    }

    public static boolean isInvalidFilePath(File path) {
        for (File node = path; node != null; node = node.getParentFile()) {
            if (!FileUtilities.isInvalidFileName(node.getName())) continue;
            return true;
        }
        return false;
    }

    public static String normalizePathSeparators(String path) {
        if (path.startsWith(UNC_PREFIX)) {
            return UNC_PREFIX + FileUtilities.normalizePathSeparators(path.substring(UNC_PREFIX.length()));
        }
        return FileUtilities.replacePathSeparators(path, "/");
    }

    public static String replacePathSeparators(CharSequence path) {
        return FileUtilities.replacePathSeparators(path, " ");
    }

    public static String replacePathSeparators(CharSequence path, String replacement) {
        return RegularExpressions.SLASH.matcher(path).replaceAll(replacement);
    }

    public static String md5(String string) {
        return FileUtilities.md5(StandardCharsets.UTF_8.encode(string));
    }

    public static String md5(byte[] data) {
        return FileUtilities.md5(ByteBuffer.wrap(data));
    }

    public static String md5(ByteBuffer data) {
        try {
            MessageDigest hash = MessageDigest.getInstance("MD5");
            hash.update(data);
            return String.format("%032x", new BigInteger(1, hash.digest()));
        }
        catch (NoSuchAlgorithmException e) {
            throw new RuntimeException(e);
        }
    }

    public static List<File> asFileList(Object ... paths) {
        ArrayList<File> files = new ArrayList<File>(paths.length);
        for (Object it : paths) {
            if (it instanceof CharSequence) {
                files.add(new File(it.toString()));
                continue;
            }
            if (it instanceof File) {
                files.add((File)it);
                continue;
            }
            if (it instanceof Path) {
                files.add(((Path)it).toFile());
                continue;
            }
            if (!(it instanceof Collection)) continue;
            files.addAll(FileUtilities.asFileList(((Collection)it).toArray()));
        }
        return files;
    }

    public static String formatSize(long size) {
        if (size >= 100000000000L) {
            return String.format("%,d GB", size / 1000000000L);
        }
        if (size >= 10000000000L) {
            return String.format("%.1f GB", (double)size / 1.0E9);
        }
        if (size >= 1000000000L) {
            return String.format("%.2f GB", (double)size / 1.0E9);
        }
        if (size >= 10000000L) {
            return String.format("%,d MB", size / 1000000L);
        }
        if (size >= 1000000L) {
            return String.format("%.1f MB", (double)size / 1000000.0);
        }
        if (size >= 1000L) {
            return String.format("%,d KB", size / 1000L);
        }
        return String.format("%,d bytes", size);
    }

    private FileUtilities() {
        throw new UnsupportedOperationException();
    }

    public static class RegexFileFilter
    implements FileFilter,
    FilenameFilter {
        private final Pattern pattern;

        public RegexFileFilter(Pattern pattern) {
            this.pattern = pattern;
        }

        @Override
        public boolean accept(File dir, String name) {
            return this.pattern.matcher(name).matches();
        }

        @Override
        public boolean accept(File file) {
            return this.accept(file.getParentFile(), file.getName());
        }
    }

    public static class ExtensionFileFilter
    implements FileFilter,
    FilenameFilter {
        public static final List<String> WILDCARD = Collections.singletonList("*");
        private final String[] extensions;

        public ExtensionFileFilter(String ... extensions) {
            this.extensions = (String[])extensions.clone();
        }

        public ExtensionFileFilter(Collection<String> extensions) {
            this.extensions = extensions.toArray(new String[0]);
        }

        @Override
        public boolean accept(File dir, String name) {
            return this.accept(name);
        }

        @Override
        public boolean accept(File file) {
            return this.accept(file.getName());
        }

        public boolean accept(String name) {
            return this.acceptAny() || FileUtilities.hasExtension(name, this.extensions);
        }

        public boolean acceptAny() {
            return this.extensions.length == 1 && WILDCARD.get(0).equals(this.extensions[0]);
        }

        public boolean acceptExtension(String extension) {
            if (this.acceptAny()) {
                return true;
            }
            for (String other : this.extensions) {
                if (!other.equalsIgnoreCase(extension)) continue;
                return true;
            }
            return false;
        }

        public String extension() {
            return this.extensions[0];
        }

        public String[] extensions() {
            return (String[])this.extensions.clone();
        }

        public String toString() {
            StringBuilder s = new StringBuilder();
            for (String it : this.extensions) {
                if (s.length() > 0) {
                    s.append(", ");
                }
                s.append("*.").append(it);
            }
            return s.toString();
        }

        public static ExtensionFileFilter union(ExtensionFileFilter ... filters) {
            ArrayList<String> extensions = new ArrayList<String>();
            for (ExtensionFileFilter it : filters) {
                if (it.acceptAny()) continue;
                Collections.addAll(extensions, it.extensions());
            }
            return new ExtensionFileFilter(extensions);
        }
    }

    public static class ParentFilter
    implements FileFilter {
        private final File folder;

        public ParentFilter(File folder) {
            this.folder = folder;
        }

        @Override
        public boolean accept(File file) {
            return FileUtilities.listPath(file).contains(this.folder);
        }
    }
}

